module srisc7(input clock, input reset, 
	output [11:0] pcx, output [31:0] ir, output [31:0] results,
	output call,jmpr
	);


reg [7:0] pc;
wire [7:0] newpc;
// make program counter look like byte-addressed memory
assign pcx = {2'b00,pc,2'b00};

// update program counter
always @(posedge clock or negedge reset)
begin
	if (!reset) pc <= 8'h0b; // so that newpc starts at 0x0c
	else pc <= newpc;
end

// instantiate memory
inst instruction_cache(
	.address(newpc),
	.clock(clock),
	.q(ir)
);


// instantiate register file

wire [31:0] da, db, dc;
wire [4:0] ra, rb, rc;
wire write_enable;

reg_file regs(
	.clock(clock),
	.reset(reset),
	.da(da),
	.db(db),
	.dc(dc),
	.ra(ra),
	.rb(rb),
	.rc(rc),
	.write_enable(write_enable)
	);

// decode instruction

wire [5:0] op, opx;
wire [31:0] immedse;
wire [4:0] shf;
assign op = ir[5:0];
assign ra = ir[31:27];
assign rb = ir[26:22];
assign opx = ir[16:11];
assign shf = ir[10:6];
assign immedse = {(ir[21]?16'hffff:16'h0000),ir[21:6]};

wire iflag;
assign iflag = (op!=6'h3a);
assign rc = (call? 5'h1f: (iflag? rb: ir[21:17]));

//--------------------------------------------------------
// instantiate r-format control unit
wire [5:0] rcontrol;
rformat unit1(
	.opx(opx),
	.control(rcontrol)
	);
	
// instantiate i-format control unit
wire [5:0] icontrol;
wire [1:0] muxc;
wire ibranch;
iformat unit2(
	.op(op),
	.control(icontrol),
	.muxc(muxc),
	.branch(ibranch)
	);

wire [5:0] control = (iflag? icontrol: rcontrol);

//----------------------------------------------------------
// instantiate input multiplexer

wire [31:0] dmuxi; // muxi output
wire [1:0] muxi_control  = (iflag? muxc: 2'b00);
muxi mux1(
	.dst(dmuxi),
	.b(db),
	.immedse(immedse),
	.control(muxi_control)
	);
//----------------------------------------------------------
// instantiate ALU

wire [31:0] dc1; // alu output
wire [2:0] cmp_control = {1'b0,control[2],1'b1};
wire do_control = (control[5:3]==3'b001);
wire [2:0] alu_control = (do_control? cmp_control: control[2:0]);

wire zero, neg, ovfl;

alu my_alu(
	.a(da),
	.b(dmuxi),
	.control(alu_control),
	.c(dc1),
	.zero(zero),
	.neg(neg),
	.ovfl(ovfl)
	);
//----------------------------------------------------------
// instantiate barrel shifter
wire [4:0] shift = (control[3]? shf: db[4:0]);
wire [31:0] dc2;

barrel brl(
	.inp(da),
	.control(control[2:0]),
	.shift(shift),
	.out(dc2)
	);
	
//----------------------------------------------------------
// include logic for compare operations
wire cmp = (control[1]? neg: zero)^control[0];


//---------------------------------------------------------
// branch logic


wire[7:0] nextpc = pc + 1'b1;
assign branch = (iflag? ibranch: 1'b0);

assign do_branch = branch&&cmp;
wire[7:0] branchpc = nextpc+(do_branch?immedse[9:2]:8'h00);

assign write_enable = !branch;

//----------------------------------------------------------
// call/jump logic

// add (|ir) so we don't allow zero-valued instruction to be interpreted as a "call"
assign call = (op==6'h00) && (|ir);
assign jmpr = (muxd==3'h7);
wire [7:0] rpc = (call? ir[13:6]:da[9:2]);
assign newpc = (call||jmpr)? rpc: branchpc;

//----------------------------------------------------------
// instantiate dc-bus multiplexer
wire [2:0] muxd = control[5:3];

muxc mux2(
	.oal(dc1),
	.obs(dc2),
	.cmp(cmp),
	.nextpc(nextpc),
	.data(ir),
	.control(muxd),
	.out(dc)
	);

assign results = dc; // also send output outside the module for display

endmodule
