module srisc6(input clock, input reset, output [11:0] pcx, output [31:0] ir, output [31:0] results);

// instantiate memory
wire [7:0] new_pc;
inst inst_cache(
	.address(new_pc),
	.clock(clock),
	.q(ir)
);

reg [7:0] pc;
// make program counter look like byte-addressed memory address
assign pcx = 8'h30 + {pc,2'b00}; 

always @(posedge clock or negedge reset)
	if (!reset) 
		pc <= 8'hff; // so that next_pc starts at zero
	else 
		pc <= new_pc;

// Instantiate register file

wire [31:0] da, db;
reg  [31:0] dc;
wire [4:0] ra, rb, rc;
wire write_enable;

reg_file regs(
	.clock(clock),
	.reset(reset),
	.da(da),
	.db(db),
	.dc(dc),
	.ra(ra),
	.rb(rb),
	.rc(rc),
	.write_enable(write_enable)
	);

// decode instruction

wire [5:0] op, opx;
wire [31:0] immedse;
wire [4:0] shf;
assign op = ir[5:0];
assign ra = ir[31:27];
assign rb = ir[26:22];
assign opx = ir[16:11];
assign shf = ir[10:6];
assign immedse = {(ir[21]?16'hffff:16'h0000),ir[21:6]};

wire iflag;
assign iflag = (op!=6'h3a);
assign rc = (iflag? rb: ir[21:17]);

//--------------------------------------------------------
// instantiate r-format control unit
wire [5:0] rcontrol;
rformat unit1(
	.opx(opx),
	.control(rcontrol)
	);
	
// instantiate i-format control unit
wire [5:0] icontrol;
wire [1:0] muxc;
wire ibranch;
iformat unit2(
	.op(op),
	.control(icontrol),
	.muxc(muxc),
	.branch(ibranch)
	);

wire [5:0] control = (iflag? icontrol: rcontrol);

//----------------------------------------------------------
// instantiate immediate/b-data multiplexer

wire [31:0] dmuxi; // muxi output
wire [1:0] muxi_control  = (iflag? muxc: 2'b00);
muxi mux1(
	.dst(dmuxi),
	.b(db),
	.immedse(immedse),
	.control(muxi_control)
	);
//----------------------------------------------------------
// instantiate ALU

wire [31:0] dc1; // alu output
wire is_compare = (control[5:3]==3'b001);
wire [2:0] cmp_control = {1'b0,control[2],1'b1};
wire [2:0] alu_control = (is_compare? cmp_control: control[2:0]);

wire zero, neg, ovfl;

alu my_alu(
	.a(da),
	.b(dmuxi),
	.control(alu_control),
	.c(dc1),
	.zero(zero),
	.neg(neg),
	.ovfl(ovfl)
	);
//----------------------------------------------------------
// instantiate barrel shifter
wire [4:0] shift = (control[3]? shf: db[4:0]);
wire [31:0] dc2;

barrel brl(
	.inp(da),
	.control(control[2:0]),
	.shift(shift),
	.out(dc2)
	);
	
//----------------------------------------------------------
// include logic for compare operations
wire cmp = (control[1]? neg: zero)^control[0];


//---------------------------------------------------------
// branch logic


wire[7:0] next_pc = pc + 1'b1;
assign branch = (iflag? ibranch: 1'b0);

assign do_branch = branch&&cmp;
assign new_pc = next_pc+(do_branch?immedse[9:2]:8'h00);

assign write_enable = !branch;

//----------------------------------------------------------
// instantiate dc-bus multiplexer
wire [2:0] muxd = control[5:3];

always
case(muxd[1:0])
	2'b00: dc <= dc1;
	2'b01: dc <= {31'h00000000,cmp};
	2'b10: dc <= dc2;
	2'b11: dc <= dc2;
	default: dc <= 32'h00000000;
endcase

assign results = dc; // also send output outside the module for display

endmodule
