module srisc4(input clock, input reset, output reg [7:0] pc, output [31:0] ir, output [31:0] results);

// instantiate memory
wire [7:0] next_pc;
inst inst_cache(
	.address(next_pc),
	.clock(clock),
	.q(ir)
);

assign next_pc = pc + 1'b1;

always @(posedge clock or negedge reset)
	if (!reset) 
		pc <= 8'hff; // so that next_pc starts at zero
	else 
		pc <= next_pc;

// Instantiate register file

wire [31:0] da, db, dc;
wire [4:0] ra, rb, rc;
wire write_enable;

reg_file regs(
	.clock(clock),
	.reset(reset),
	.da(da),
	.db(db),
	.dc(dc),
	.ra(ra),
	.rb(rb),
	.rc(rc),
	.write_enable(write_enable)
	);

// decode instruction

wire [5:0] op, opx;
wire [31:0] immedse;
wire [4:0] shf;
assign op = ir[5:0];
assign ra = ir[31:27];
assign rb = ir[26:22];
assign opx = ir[16:11];
assign shf = ir[10:6];
assign immedse = {(ir[21]?16'hffff:16'h0000),ir[21:6]};

wire iflag;
assign iflag = (op!=6'h3a);
assign rc = (iflag? rb: ir[21:17]);

//--------------------------------------------------------
// instantiate r-format control unit
wire [5:0] rcontrol;
rformat unit1(
	.opx(opx),
	.control(rcontrol)
	);
	
// instantiate i-format control unit
wire [5:0] icontrol;
wire [1:0] muxc;
iformat unit2(
	.op(op),
	.control(icontrol),
	.muxc(muxc)
	);

wire [5:0] control = (iflag? icontrol: rcontrol);

//----------------------------------------------------------
// instantiate immediate/b-data multiplexer

wire [31:0] dmuxi; // muxi output
wire [1:0] muxi_control  = (iflag? muxc: 2'b00);
muxi mux1(
	.dst(dmuxi),
	.b(db),
	.immedse(immedse),
	.control(muxi_control)
	);
//----------------------------------------------------------
// instantiate ALU

wire [31:0] dc1; // alu output
wire [2:0] alu_control = (control[4:3]==(2'b01)? {1'b0,control[2],1'b1}: control[2:0]);

wire zero, ovfl;

alu my_alu(
	.a(da),
	.b(dmuxi),
	.control(alu_control),
	.c(dc1),
	.zero(zero),
	.ovfl(ovfl)
	);
//----------------------------------------------------------
// instantiate barrel shifter
wire [4:0] shift = (control[3]? shf: db[4:0]);
wire [31:0] dc2;

barrel brl(
	.inp(da),
	.control(control[2:0]),
	.shift(shift),
	.out(dc2)
	);
	


//----------------------------------------------------------
// instantiate dc-bus multiplexer
wire [2:0] muxd = control[5:3];

assign dc = (muxd[1]? dc2: dc1);

assign write_enable = 1'b1;
assign results = dc; // also send output outside the module for display

endmodule
