module dcpu2 
#(parameter DATA_WIDTH=16, parameter ADDR_WIDTH=8)
(
	input clk, reset_n, loopback,
	output reg [7:0] pc,
	// memory bus
	output [(ADDR_WIDTH-1):0] addr,
	input [(DATA_WIDTH-1):0] q,
	output [(DATA_WIDTH-1):0] data,
	output we
);

wire ma, rwe;
wire [2:0] ra, rb, rd;
wire [(DATA_WIDTH-1):0] qa, qb, qd, qi;
assign data = qb;

regfile dev2(
	.clk(clk),
	.ra(ra), .rb(rb), .rd(rd),
	.qa(qa), .qb(qb), .qd(qd),
	.we(rwe)
);

//reg [7:0] pc;
assign rwe = state==WB;


wire [4:0] opcode;

reg [15:0] inst;
assign qi = (state==ID? q: inst);
assign opcode = qi[15:11];
wire [7:0] immed = qi[7:0];
	
reg [2:0] state;
localparam 
	IF = 3'h0,
	ID = 3'h1,
	EX = 3'h2,
	MEM = 3'h3,
	WB = 3'h4,
	HALT = 3'h7;
	
`include "opcodes.txt"

wire op_sw = opcode==SW || opcode==SWI;
wire op_lw = opcode==LW || opcode==LWI;
wire op_lui = opcode==LUI;
wire op_bne = opcode==BNE;
wire op_jalr = opcode==JALR;

assign we = (state==MEM) && op_sw;
assign ma = (op_sw || op_lw); // memory_access

assign rd = (op_sw||op_bne? 3'h0: qi[10:8]);
assign ra = (op_lui||op_bne? qi[10:8]: qi[7:5]);
assign rb = (op_sw? qi[10:8]: op_bne? qi[7:5]: qi[4:2]);

wire do_branch = op_bne && (qa!=qb);

reg [7:0] jump;
wire [7:0] nextpc = pc + 1'b1;
wire [7:0] newpc = (op_jalr||do_branch? jump: nextpc);

assign addr = (state==MEM? qt[7:0]: newpc);

// ALU
wire [15:0] qt;
alu dev3(
.a(qa),
.b(qb),
.nextpc(nextpc),
.imm8(immed),
.d(qt),
.opcode(opcode)
);

// writeback register multiplexer
assign qd = (op_lw? q: qt);


always @(posedge clk, negedge reset_n)
begin
  if (!reset_n)
  begin
   pc <= 8'hFF;
   state <= IF;
   end
  else 
	case (state)
	IF:
	begin
		state <= ID;
		pc <= addr;
    end
    ID:
    begin
	 // save results of IF phase
	 inst <= q;
	 state <= EX;
	end
	EX:
	begin
	  jump <= op_jalr? qa[7:0]: qt[7:0];
	  if (inst == 16'h0000) state <= HALT;
      else if (ma)
		state <= MEM;
      else 
		state <= WB;
	end
   MEM:
    begin
	   state <= WB;
	end
	WB:
	begin
	   state <= IF;
	end
	HALT:
	begin
		if (loopback)
		begin
			pc <= 8'hFF;
			state <= IF;
		end
		else state <= HALT;
	end
	default:
		state <= HALT;
  endcase
end


endmodule